<?php
/**
 * Bolt magento2 plugin
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 *
 * @category   Bolt
 * @package    Bolt_Boltpay
 * @copyright  Copyright (c) 2017-2023 Bolt Financial, Inc (https://www.bolt.com)
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

namespace Bolt\Boltpay\ThirdPartyModules\Amasty;

use Amasty\GiftCardAccount\Api\Data\GiftCardAccountInterface;
use Amasty\GiftCardAccount\Model\GiftCardAccount\GiftCardCartProcessor;
use Amasty\GiftCardAccount\Model\OptionSource\AccountStatus;
use Bolt\Boltpay\Helper\Bugsnag;
use Bolt\Boltpay\Helper\Discount;
use Bolt\Boltpay\Helper\Shared\CurrencyUtils;
use Magento\Framework\App\ResourceConnection;
use Magento\Quote\Model\Quote;
use Magento\Sales\Model\Order;
use Bolt\Boltpay\Helper\FeatureSwitch\Decider;

class GiftCardAccount
{

    /**
     * @var Bugsnag Bugsnag helper instance
     */
    private $bugsnagHelper;

    /**
     * @var Discount
     */
    private $discountHelper;

    /**
     * @var ResourceConnection
     */
    private $resourceConnection;
    
    /**
     * @var Bolt\Boltpay\Helper\FeatureSwitch\Decider
     */
    private $featureSwitches;

    /**
     * @param Bugsnag            $bugsnagHelper Bugsnag helper instance
     * @param Discount           $discountHelper
     * @param ResourceConnection $resourceConnection
     * @param Decider            $featureSwitches
     */
    public function __construct(
        Bugsnag $bugsnagHelper,
        Discount $discountHelper,
        ResourceConnection $resourceConnection,
        Decider $featureSwitches
    ) {
        $this->bugsnagHelper = $bugsnagHelper;
        $this->discountHelper = $discountHelper;
        $this->resourceConnection = $resourceConnection;
        $this->featureSwitches = $featureSwitches;
    }

    /**
     * Restores Amasty Giftcard balances used in an order that is going to be deleted
     *
     * @param \Amasty\GiftCardAccount\Model\GiftCardAccount\Repository         $giftcardRepository
     * @param \Amasty\GiftCardAccount\Model\GiftCardExtension\Order\Repository $giftcardOrderRepository
     * @param Order                                                            $order
     */
    public function beforeFailedPaymentOrderSave($giftcardRepository, $giftcardOrderRepository, $order)
    {
        try {
            $giftcardOrderExtension = $giftcardOrderRepository->getByOrderId($order->getId());
            foreach ($giftcardOrderExtension->getGiftCards() as $orderGiftcard) {
                try {
                    /** @see GiftCardCartProcessor::GIFT_CARD_ID */
                    $giftcard = $giftcardRepository->getById($orderGiftcard['id']);
                    $giftcard->setCurrentValue(
                        /** @see GiftCardCartProcessor::GIFT_CARD_BASE_AMOUNT */
                        (float)($giftcard->getCurrentValue() + $orderGiftcard['b_amount'])
                    );
                    /** @see \Amasty\GiftCardAccount\Model\OptionSource\AccountStatus::STATUS_ACTIVE */
                    $giftcard->setStatus(1);
                    $giftcardRepository->save($giftcard);
                } catch (\Magento\Framework\Exception\LocalizedException $e) {
                    $this->bugsnagHelper->notifyException($e);
                }
            }
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            //no giftcards applied on order, safe to ignore
        }
    }

    /**
     * @param array                                                          $result
     * @param \Amasty\GiftCardAccount\Api\GiftCardAccountRepositoryInterface $giftcardAccountRepository
     * @param \Amasty\GiftCardAccount\Api\GiftCardQuoteRepositoryInterface   $giftcardQuoteRepository
     * @param \Magento\Quote\Model\Quote                                     $quote
     * @param \Magento\Quote\Model\Quote                                     $parentQuote
     * @param bool                                                           $paymentOnly
     * @return array
     */
    public function collectDiscounts(
        $result,
        $giftcardAccountRepository,
        $giftcardQuoteRepository,
        $quote,
        $parentQuote,
        $paymentOnly
    ) {
        list ($discounts, $totalAmount, $diff) = $result;

        try {
            $currencyCode = $quote->getQuoteCurrencyCode();
            /** @var \Magento\Quote\Model\Quote\Address\Total[] */
            $totals = $quote->getTotals();
            $totalDiscount = $totals[Discount::AMASTY_GIFTCARD] ?? null;
            $roundedDiscountAmount = 0;
            $discountAmount = 0;
            ///////////////////////////////////////////////////////////////////////////
            // If Amasty gift cards can be used for shipping and tax (PayForEverything)
            // accumulate all the applied gift cards balance as discount amount. If the
            // final discounts sum is greater than the cart total amount ($totalAmount < 0)
            // the "fixed_amount" type is added below.
            ///////////////////////////////////////////////////////////////////////////
            if ($totalDiscount && $totalDiscount->getValue()) {
                $appliedGiftCardItems = [];
                if ($this->discountHelper->getAmastyPayForEverything()) {
                    $giftcardQuote = $giftcardQuoteRepository->getByQuoteId($quote->getId());
                    foreach ($giftcardQuote->getGiftCards() as $appliedGiftcardData) {
                        $giftcard = $giftcardAccountRepository->getById($appliedGiftcardData['id']);
                        $amount = abs((float)$giftcard->getCurrentValue());               
                        $giftCardCode = $giftcard->getCodeModel()->getCode();
                        $appliedGiftCardItems[] = [
                            'code' => $giftCardCode,
                            'amount' => $amount
                        ];
                    }
                } else {
                    $extension = $quote->getExtensionAttributes();
                    $appliedGiftCards = $extension->getAmAppliedGiftCards();
                    foreach ($appliedGiftCards as $appliedGiftCard) {
                        $amount = abs((float)$appliedGiftCard[\Amasty\GiftCardAccount\Model\GiftCardAccount\GiftCardCartProcessor::GIFT_CARD_AMOUNT]);               
                        $giftCardCode = $appliedGiftCard[\Amasty\GiftCardAccount\Model\GiftCardAccount\GiftCardCartProcessor::GIFT_CARD_CODE];
                        $appliedGiftCardItems[] = [
                            'code' => $giftCardCode,
                            'amount' => $amount
                        ];
                    }
                }
                foreach ($appliedGiftCardItems as $appliedGiftCardItem) {
                    $roundedAmount = CurrencyUtils::toMinor($appliedGiftCardItem['amount'], $currencyCode);
                    $discountItem = [
                        'description'       => __('Gift Card ') . $appliedGiftCardItem['code'],
                        'amount'            => $roundedAmount,
                        'discount_category' => Discount::BOLT_DISCOUNT_CATEGORY_GIFTCARD,
                        'reference'         => $appliedGiftCardItem['code'],
                        'discount_type'     => Discount::BOLT_DISCOUNT_TYPE_FIXED, // For v1/discounts.code.apply and v2/cart.update
                        'type'              => Discount::BOLT_DISCOUNT_TYPE_FIXED, // For v1/merchant/order
                    ];
                    $discountAmount += $appliedGiftCardItem['amount'];
                    $roundedDiscountAmount += $roundedAmount;
                    $discounts[] = $discountItem;
                }

                $diff -= CurrencyUtils::toMinorWithoutRounding($discountAmount, $currencyCode) - $roundedDiscountAmount;
                $totalAmount -= $roundedDiscountAmount;
            }
        } catch (\Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        } finally {
            return [$discounts, $totalAmount, $diff];
        }
    }

    /**
     * @param null                                                           $result
     * @param \Amasty\GiftCardAccount\Api\GiftCardAccountRepositoryInterface $giftcardAccountRepository
     * @param string                                                         $couponCode
     * @param Quote                                                          $quote
     * @return GiftCardAccountInterface|null
     */
    public function loadGiftcard($result, $giftcardAccountRepository, $couponCode, $quote)
    {
        if ($result !== null) {
            return $result;
        }
        
        try {
            return $giftcardAccountRepository->getByCode($couponCode);
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            return null;
        } catch (\Exception $e) {
            if ($this->featureSwitches->isReturnErrWhenRunFilter()) {
                return $e;
            } else {
                $this->bugsnagHelper->notifyException($e);
                return null;
            }
        }
    }

    /**
     * @see \Bolt\Boltpay\Model\Api\DiscountCodeValidation::applyingGiftCardCode
     *
     *
     * @param null                                                                $result
     * @param \Amasty\GiftCardAccount\Model\GiftCardAccount\GiftCardCartProcessor $giftcardProcessor
     * @param string                                                              $code
     * @param GiftCardAccountInterface                                            $giftCard
     * @param Quote                                                               $immutableQuote
     * @param Quote                                                               $parentQuote
     * @return array
     */
    public function applyGiftcard(
        $result,
        $giftcardProcessor,
        $code,
        $giftCard,
        $immutableQuote,
        $parentQuote
    ) {
        if (!$giftCard instanceof GiftCardAccountInterface) {
            return $result;
        }
        try {
            foreach ([$parentQuote, $immutableQuote] as $quote) {
                $isGiftcardApplied = !empty(
                    array_filter(
                        $quote->getExtensionAttributes()->getAmGiftcardQuote()->getGiftCards(),
                        function ($giftCardData) use ($giftCard) {
                            return $giftCard->getAccountId() == $giftCardData['id'];
                        }
                    )
                );
                if ($isGiftcardApplied) {
                    continue;
                }
                $giftcardProcessor->applyToCart($giftCard, $quote);
            }

            return [
                'status'          => 'success',
                'discount_code'   => $code,
                'discount_amount' => abs(
                    CurrencyUtils::toMinor($giftCard->getCurrentValue(), $parentQuote->getQuoteCurrencyCode())
                ),
                'description'     => __('Gift Card (%1)', $code),
                'discount_type'   => 'fixed_amount',
            ];
        } catch (\Exception $e) {
            return [
                'status'        => 'failure',
                'error_message' => $e->getMessage(),
            ];
        }
    }

    /**
     * @param bool                           $result
     * @param GiftCardCartProcessor          $giftcardProcessor
     * @param string                         $couponCode
     * @param mixed|GiftCardAccountInterface $giftCard
     * @param Quote                          $quote
     * @return bool
     */
    public function filterApplyingGiftCardCode(
        $result,
        $giftcardProcessor,
        $couponCode,
        $giftCard,
        $quote
    ) {
        if (!$giftCard instanceof GiftCardAccountInterface) {
            return $result;
        }
        try {
            $giftcardProcessor->applyToCart($giftCard, $quote);
            $result = true;
        } catch (\Magento\Framework\Exception\LocalizedException $e) {
        }
        return $result;
    }

    /**
     * @param bool                     $result
     * @param GiftCardCartProcessor    $giftcardProcessor
     * @param GiftCardAccountInterface $giftCard
     * @param Quote                    $quote
     *
     * @return bool
     */
    public function filterRemovingGiftCardCode(
        $result,
        $giftcardProcessor,
        $giftCard,
        $quote
    ) {
        if (!$giftCard instanceof GiftCardAccountInterface) {
            return $result;
        }
        try {
            $giftcardProcessor->removeFromCart($giftCard, $quote);
            $result = true;
        } catch (\Magento\Framework\Exception\LocalizedException $e) {
        }
        return $result;
    }

    /**
     * @param Quote $source
     * @param Quote $destination
     */
    public function replicateQuoteData(
        $source,
        $destination
    ) {
        if ($source->getId() == $destination->getId()) {
            return;
        }
        $connection = $this->resourceConnection->getConnection();
        $connection->beginTransaction();
        $giftCardTable = $this->resourceConnection->getTableName('amasty_giftcard_quote');

        // Clear previously applied gift cart codes from the immutable quote
        $sql = "DELETE FROM {$giftCardTable} WHERE quote_id = :destination_quote_id";
        $connection->query($sql, ['destination_quote_id' => $destination->getId()]);

        // Copy all gift cart codes applied to the parent quote to the immutable quote
        $sql = "INSERT INTO {$giftCardTable} (quote_id, gift_cards, gift_amount, base_gift_amount, gift_amount_used, base_gift_amount_used)
                        SELECT :destination_quote_id, gift_cards, gift_amount, base_gift_amount, gift_amount_used, base_gift_amount_used
                        FROM {$giftCardTable} WHERE quote_id = :source_quote_id";

        $connection->query(
            $sql,
            ['destination_quote_id' => $destination->getId(), 'source_quote_id' => $source->getId()]
        );

        $connection->commit();
    }
    
    /**
     * @param Quote $quote
     */
    public function clearExternalData($quote)
    {
        try {
            $connection = $this->resourceConnection->getConnection();
            $giftCardTable = $this->resourceConnection->getTableName('amasty_giftcard_quote');

            $sql = "DELETE FROM {$giftCardTable} WHERE quote_id = :quote_id";
            $bind = [
                'quote_id' => $quote->getId()
            ];

            $connection->query($sql, $bind);
        } catch (\Zend_Db_Statement_Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        }
    }
    
    /**
     * @param Quote $quote
     */
    public function deleteRedundantDiscounts($quote)
    {
        try {
            $connection = $this->resourceConnection->getConnection();
            $giftCardTable = $this->resourceConnection->getTableName('amasty_giftcard_quote');
            $quoteTable = $this->resourceConnection->getTableName('quote');

            $sql = "DELETE FROM {$giftCardTable} WHERE quote_id IN 
                    (SELECT entity_id FROM {$quoteTable} 
                    WHERE bolt_parent_quote_id = :bolt_parent_quote_id AND entity_id != :entity_id)";
            
            $bind = [
                'bolt_parent_quote_id' => $quote->getBoltParentQuoteId(),
                'entity_id' => $quote->getBoltParentQuoteId()
            ];

            $connection->query($sql, $bind);
        } catch (\Zend_Db_Statement_Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        }
    }
    
    /**
     * Remove Amasty Gift Card and update quote totals
     *
     * @param int $codeId
     * @param Quote $quote
     */
    public function removeAmastyGiftCard($amastyGiftCardAccountManagement, $codeId, $quote)
    {
        try {
            if (!$quote->getExtensionAttributes() || !$quote->getExtensionAttributes()->getAmGiftcardQuote()) {
                return;
            }
            $cards = $quote->getExtensionAttributes()->getAmGiftcardQuote()->getGiftCards();
            $giftCodeExists = false;
            $giftCode = '';
            foreach ($cards as $k => $card) {
                if ($card['id'] == $codeId) {
                    $giftCodeExists = true;
                    $giftCode = $card['code'];
                    break;
                }
            }
            
            if ($giftCodeExists) {
                $amastyGiftCardAccountManagement->removeGiftCardFromCart($quote->getId(), $giftCode);
            }
        } catch (\Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        }
    }
    
    /**
     * Clone the gift card info to the latest updated order.
     *
     * @param \Magento\Sales\Model\Order $result
     * @param \Amasty\GiftCardAccount\Model\GiftCardExtension\Order\Handlers\SaveHandler $giftCardAccountRepository
     * @param \Magento\Sales\Model\Order $order
     */
    public function beforeGetOrderByIdProcessNewOrder($result, $giftCardAccountSaveHandler, $order)
    {
        try {
            if ($order->getExtensionAttributes() && $order->getExtensionAttributes()->getAmGiftcardOrder()) {
                $gCardOrder = $order->getExtensionAttributes()->getAmGiftcardOrder();
                $existingOrderExtension = $result->getExtensionAttributes();
                $gCardOrder->setOrderId((int)$result->getId());
                $existingOrderExtension->setAmGiftcardOrder($gCardOrder);
                $result->setExtensionAttributes($existingOrderExtension);
                $giftCardAccountSaveHandler->saveAttributes($result);
            }
        } catch (\Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        } finally {
            return $result;
        }
    }
    
    /**
     * Complete the transaction of Amasty Gift Card once the order is created.
     *
     * @param \Amasty\GiftCardAccount\Model\GiftCardAccount\GiftCardAccountTransactionProcessor $giftCardAccountTransactionProcessor
     * @param \Amasty\GiftCardAccount\Model\GiftCardAccount\Repository $giftCardAccountRepository
     * @param \Magento\Sales\Model\Order $order
     * @param Quote $quote
     * @param \stdClass $transaction
     */
    public function orderPostprocess($giftCardAccountTransactionProcessor, $giftCardAccountRepository, $order, $quote, $transaction)
    {
        if (!$order->getExtensionAttributes() || !$order->getExtensionAttributes()->getAmGiftcardOrder()) {
            return;
        }
        /** @var GiftCardOrderInterface $gCardOrder */
        $gCardOrder = $order->getExtensionAttributes()->getAmGiftcardOrder();
        try {
            foreach ($gCardOrder->getAppliedAccounts() as $appliedAccount) {
                $giftCardAccountRepository->save($appliedAccount);
                $giftCardAccountTransactionProcessor->completeTransaction($appliedAccount);
            }
        } catch (\Exception $e) {
            $this->bugsnagHelper->notifyException($e);
        }
    }
}
