<?php
/**
 * Bolt magento2 plugin
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Open Software License (OSL 3.0)
 * that is bundled with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://opensource.org/licenses/osl-3.0.php
 *
 * @category   Bolt
 * @package    Bolt_Boltpay
 * @copyright  Copyright (c) 2017-2023 Bolt Financial, Inc (https://www.bolt.com)
 * @license    http://opensource.org/licenses/osl-3.0.php  Open Software License (OSL 3.0)
 */

namespace Bolt\Boltpay\Plugin\Taxjar\SalesTax\Model;

use Magento\Quote\Api\Data\ShippingAssignmentInterface;
use Magento\Quote\Model\Quote;
use Magento\Tax\Api\Data\QuoteDetailsInterface;
use Taxjar\SalesTax\Model\Smartcalcs;
use Magento\Framework\Pricing\PriceCurrencyInterface;

class SmartcalcsPlugin
{
    /**
     * @var PriceCurrencyInterface
     */
    private $priceCurrency;

    public function __construct(PriceCurrencyInterface $priceCurrency)
    {
        $this->priceCurrency = $priceCurrency;
    }

    /**
     * Magento rounds prices to 2 decimal points.
     * ref: https://docs.magento.com/user-guide/tax/quick-reference.html,
     * https://github.com/magento/magento2/blob/2.4/app/code/Magento/Quote/Model/Quote/Item/AbstractItem.php#L407
     *
     * On the contrary, TaxJar doesn’t do any rounding
     * ref: https://github.com/taxjar/taxjar-magento2-extension/blob/master/Model/Tax/TaxCalculation.php#L201
     *
     * For instance, a product has a price of $32.66 and a tier discount (quantity 5 or more) of 10%
     * which equals to $29,394. When calculating for quantity of 10, Magento calculation equals to $293.90
     * (rounded before multiplication). But after triggering Taxjar (by entering an address) the subtotal
     * becomes $293.94 (not rounded before multiplication).
     *
     * What TaxJar does is that, when in the total calculation pipeline, it rewrites the previous (subtotal)
     * values generated by Magento.
     *
     * This can cause problems with our plugin which relies on 2 decimal points rounding,
     * as used and recommended by Magento.
     *
     * Namely, after the cart data is collected and sent to Bolt the subtotal may change after tax processing
     * which leads to order total mismatch.
     *
     * @param Smartcalcs $subject
     * @param Quote                             $quote
     * @param QuoteDetailsInterface             $quoteTaxDetails
     * @param ShippingAssignmentInterface       $shippingAssignment
     *
     * @return array
     */
    public function beforeGetTaxForOrder(
        Smartcalcs $subject,
        Quote $quote,
        QuoteDetailsInterface $quoteTaxDetails,
        ShippingAssignmentInterface $shippingAssignment
    ) {
        foreach ($quoteTaxDetails->getItems() as $quoteDetailsItem) {
            $quoteDetailsItem->setUnitPrice(
                $this->priceCurrency->round($quoteDetailsItem->getUnitPrice())
            );
        }
        foreach ($shippingAssignment->getItems() as $shippingAssignmentItem) {
            $shippingAssignmentItem->setBaseCalculationPrice(
                $this->priceCurrency->round($shippingAssignmentItem->getBaseTaxCalculationPrice())
            );
            $shippingAssignmentItem->setTaxCalculationPrice(
                $this->priceCurrency->round($shippingAssignmentItem->getTaxCalculationPrice())
            );
            $shippingAssignmentItem->setBaseTaxCalculationPrice(
                $this->priceCurrency->round($shippingAssignmentItem->getBaseTaxCalculationPrice())
            );
        }
        return [$quote, $quoteTaxDetails, $shippingAssignment];
    }
}
